require('dotenv').config()
const express = require('express')
const cors = require('cors')
const morgan = require('morgan')
const { upsertUserFromGoogle, userExistsByUid, selectById } = require('./db')
const { verifyGoogleIdToken, issueJwt, auth } = require('./auth')
const { authUrl, handleCallback } = require('./oauth')
const gmailRoutes = require('./routes/gmail')
const driveRoutes = require('./routes/drive')

const app = express()
app.use(cors())
app.use(express.json())
app.use(morgan('tiny'))
app.use(express.static('public'))

app.get('/health', (req, res) => {
  res.json({ ok: true })
})

app.get('/config.js', (req, res) => {
  res.type('application/javascript').send(`window.GOOGLE_OAUTH_CLIENT_ID = '${process.env.GOOGLE_OAUTH_CLIENT_ID || ''}'`)
})

app.get('/user-exists', (req, res) => {
  const uid = req.query.uid
  if (!uid) return res.status(400).json(false)
  res.json(userExistsByUid(uid))
})

app.post('/auth/google', async (req, res) => {
  try {
    const idToken = req.body && req.body.idToken
    if (!idToken) return res.status(400).json({ error: 'missing_id_token' })
    console.debug('[Auth] Verifying Google ID token...')
    const payload = await verifyGoogleIdToken(idToken)
    console.debug('[Auth] Google ID token verified for:', payload.email)
    const user = upsertUserFromGoogle(payload)
    const accessToken = issueJwt(user)
    res.json({ accessToken, user: { id: user.id, email: user.email, name: user.name, avatar_url: user.avatar_url } })
  } catch (e) {
    console.error('[Auth] Google ID token verification failed:', e.message)
    res.status(401).json({ error: e.message.includes('Access blocked') ? 'organization_access_denied' : 'invalid_google_token' })
  }
})

app.get('/me', auth, (req, res) => {
  const u = selectById.get(req.user.uid)
  if (!u) return res.status(404).json({ error: 'not_found' })
  res.json({ id: u.id, email: u.email, name: u.name, avatar_url: u.avatar_url })
})

app.get('/oauth/google/start', auth, (req, res) => {
  const scopes = req.query.scope ? [].concat(req.query.scope) : [
    'https://www.googleapis.com/auth/gmail.modify',
    'https://www.googleapis.com/auth/drive'
  ]
  const url = authUrl(scopes, issueJwt({ id: req.user.uid, email: req.user.email || '' }))
  res.json({ url })
})

app.get('/oauth/google/callback', async (req, res) => {
  try {
    const code = req.query.code
    const state = req.query.state
    if (!code) return res.status(400).json({ error: 'missing_code' })
    if (!state) return res.status(400).json({ error: 'missing_state' })
    let uid
    try {
      const payload = require('jsonwebtoken').verify(state, process.env.APP_JWT_SECRET)
      uid = payload.uid || payload.id
    } catch (e) {
      console.warn('[OAuth] Invalid state JWT:', e.message)
      return res.status(401).json({ error: 'invalid_state' })
    }
    console.debug('[OAuth] Handling callback for user:', uid)
    const row = await handleCallback(uid, code)
    res.json({ ok: true, provider: 'google', scope: row.scope })
  } catch (e) {
    console.error('[OAuth] Callback failed:', e.message)
    res.status(401).json({ error: e.message.includes('Access blocked') ? 'organization_access_denied' : 'oauth_failed' })
  }
})

app.use('/gmail', gmailRoutes)
app.use('/drive', driveRoutes)

const port = process.env.PORT || 3000
app.listen(port, () => {
  process.stdout.write(`listening:${port}`)
})
