const express = require('express')
const multer = require('multer')
const { withGoogle } = require('../google')
const { auth } = require('../auth')

const router = express.Router()
const upload = multer({ storage: multer.memoryStorage(), limits: { fileSize: 25 * 1024 * 1024 } })

router.get('/files', auth, async (req, res) => {
  try {
    const q = req.query.q || ''
    const pageToken = req.query.pageToken || undefined
    const pageSize = parseInt(req.query.pageSize || '20', 10)
    const out = await withGoogle(req.user.uid, async ({ drive }) => {
      const r = await drive.files.list({
        q,
        pageToken,
        pageSize,
        fields: 'nextPageToken, files(id, name, mimeType, size, webViewLink, webContentLink, modifiedTime, owners, parents)'
      })
      return r.data
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.get('/files/:id', auth, async (req, res) => {
  try {
    const out = await withGoogle(req.user.uid, async ({ drive }) => {
      const r = await drive.files.get({ fileId: req.params.id, fields: 'id, name, mimeType, size, webViewLink, webContentLink, thumbnailLink' })
      return r.data
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.get('/files/:id/download', auth, async (req, res) => {
  try {
    await withGoogle(req.user.uid, async ({ drive }) => {
      const r = await drive.files.get({ fileId: req.params.id, alt: 'media' }, { responseType: 'stream' })
      res.setHeader('Content-Type', r.headers['content-type'] || 'application/octet-stream')
      r.data.on('error', err => res.status(500).end())
      r.data.pipe(res)
    })
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.post('/upload', auth, upload.single('file'), async (req, res) => {
  try {
    const parent = req.body.parent || undefined
    const name = req.body.name || (req.file && req.file.originalname)
    const mimeType = req.file && req.file.mimetype
    const out = await withGoogle(req.user.uid, async ({ drive }) => {
      const r = await drive.files.create({
        requestBody: { name, parents: parent ? [parent] : undefined },
        media: { mimeType, body: Buffer.from(req.file.buffer) }
      })
      return r.data
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.post('/files', auth, async (req, res) => {
  try {
    const name = req.body.name || ''
    const mimeType = req.body.mimeType || 'application/vnd.google-apps.folder'
    const parent = req.body.parent || undefined
    const out = await withGoogle(req.user.uid, async ({ drive }) => {
      const r = await drive.files.create({ requestBody: { name, mimeType, parents: parent ? [parent] : undefined } })
      return r.data
    })
    res.json(out)
  } catch (e) {
    maybeRevoke(req, e)
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.delete('/files/:id', auth, async (req, res) => {
  try {
    const out = await withGoogle(req.user.uid, async ({ drive }) => {
      await drive.files.delete({ fileId: req.params.id })
      return { ok: true }
    })
    res.json(out)
  } catch (e) {
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

router.post('/files/:id/permissions', auth, async (req, res) => {
  try {
    const type = req.body.type || 'user'
    const role = req.body.role || 'reader'
    const emailAddress = req.body.emailAddress || undefined
    const out = await withGoogle(req.user.uid, async ({ drive }) => {
      const r = await drive.permissions.create({ fileId: req.params.id, requestBody: { type, role, emailAddress }, sendNotificationEmail: false })
      return r.data
    })
    res.json(out)
  } catch (e) {
    res.status(statusFromError(e)).json(errorBody(e))
  }
})

function statusFromError(e) {
  const code = e.code || e.status || 500
  if (code === 429) return 429
  return typeof code === 'number' ? code : 500
}

function errorBody(e) {
  return { error: (e && e.message) || 'error' }
}

function maybeRevoke(req, e) {
  const code = e && (e.code || e.status)
  const msg = (e && e.message) || ''
  if (code === 401 && msg.includes('invalid_grant')) {
    console.warn('[Drive] Revoking tokens due to invalid_grant for user:', req.user.uid)
    try { require('../db').revokeOauthToken(req.user.uid, 'google') } catch (_) {}
  }
}

module.exports = router
